
/****************************************************************************
 *
 * MODULE:       r.fspl
 * AUTHOR(S):    Andrej Vilhar, Jozef Stefan Institute                
 *               Igor Ozimek (modifications & corrections), Jozef Stefan Institute
 *
 * PURPOSE:      Calculates free space loss from a given transmission point
 *               to all the other points in input DEM. 
 *             
 *
 * COPYRIGHT:    (C) 2009-2018 Jozef Stefan Institute
 *               This program is free software under the GNU General Public
 *               License (>=v2). Read the file COPYING that comes with RaPlaT
 *               for details.
 *
 *****************************************************************************/

/* History:

23-sep-2013 (I.O.)
 - minor modification

2-oct-2013 (I.O.)
 - corrected one pixel shift (right- and downwards) in the output map
 - removed redundant code
 - C source style cleaned-up

30-may-2014 (I.O.)
 - additional parameter rx_ant_height (default = 1.5)
 - additional parameters to modify freespace square-law fading:
    - exp: exponent value (default 2.0)
    - offset: constant value added to the fading (default 0.0)

15-july-2014 (I.O.)
 - skipped pathloss computation at the transmitter location (set to undefined)
   (previously produced divide-by-zero if the receive and the transmit antenna heights were equal)

1-aug-2017 (I.O.)
 - modified for GRASS GIS 7 (was GRASS GIS 6)
     GRASS API (function names, ...)
     option name changed: inputDEM -> input_dem
 - minor corrections (to remove compiler warnings, redundant code, etc.)

14-aug-2017
 - do_profile.c: changed "(DistTmp <= DistBS2MS)" to "(DistTmp < DistBS2MS)"
*/


/*
  Index space / geographic space explanation
  (generallly valid for all modules using do_profile, e.g. hataDEM)
  
  One pixel in index space corresponds to a square geographic area of geographic resolution size (e.g. 5x5m for resolution of 5m)

  Index values can be of integer or floating point type, in both cases integer values (with zero decimal part) correspond to the centre
  of the geographic square for each pixel.
  Floating point index values can be converterd to integer values (e.g. for referencing the raster buffer) by rounding.

  Indices start at the upper left corner (north-west), running rightwards along each line (x coordinate), and downwards (y coordinate).
  The upper-leftmost pixel has indeces (0, 0), the correspondig geograhic square area indeces extent is ( -0.5 .. +0.5, -0.5 .. +0.5),
  with the square center at (0.0, 0.0).
  
  Usually, the base station indeces are floating point (as converted from the transmitter geographic coordinates), while recevier location
  (mobile station) indeces are integer (generated by a "for" loop and running over the whole raster).
*/


/*
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <grass/gis.h>
#include <grass/raster.h>
#include <grass/glocale.h>
#include <math.h>
*/

#include "local_proto.h"


struct StructFSPL
{
  double BSxIndex;     /* normalized position of BS -> UTMx/resolution */
  double BSyIndex;     /* normalized position of BS -> UTMx/resolution */
  double BSAntHeight;  /* Antenna height of BS [m] */
  double MSAntHeight;  /* Antenna height of MS [m] */
  int xN;              /* dimension of teh input(Raster) and output (PathLoss) */
  int yN;              /* dimension of teh input(Raster) and output (PathLoss) */
  double scale;        /* Resolution of DEM file */
  double freq;         /* Carrier frequency in MHz */
  double ResDist;      /* Resolution model 9999 profile calc */
  double radi;         /* Radius of calculation [km] */
};

int FSPL_PathLossSub( double**, double**, struct StructFSPL, double loss_exp, double loss_offset);


/*
 * main function
 */
int main(int argc, char *argv[])
{
  double east;
  double north;
  double ant_height, frequency, radius;  /* dem_height, dem_map_LOS */
//  double rec_height = 1.5;
  double rec_height;
  double loss_exp, loss_offset;

  struct Cell_head window;  /* database window */
  struct Cell_head cellhd;  /* it stores region information and header information of rasters */
  char *name;  /* input raster name */
  char *result;  /* output raster name */
  const char *mapset;  /* mapset name */
  void *inrast;  /* input buffer */
  unsigned char *outrast;  /* output buffer */
  int nrows, ncols;
  int row, col;
  int tr_row, tr_col;
  int infd, outfd;  /* file descriptor */
  int verbose;
  struct History history;  /* holds meta-data (title, comments,..) */

  struct GModule *module;  /* GRASS module for parsing arguments */

  struct Option *input1, *opt1, *opt2, *opt3, *opt5, *opt6, *opt7, *opt8, *output;  /* options */
  struct Flag *flag1;  /* flags */

  /* initialize GIS environment */
  G_gisinit(argv[0]);  /* reads grass env, stores program name to G_program_name() */

  /* initialize module */
  module = G_define_module();
  G_add_keyword(_("raster"));
  G_add_keyword(_("fspl"));
  module->description = _("RaPlaT - Fspl module (v14aug2017)");

  /* Define the different options as defined in gis.h */
  input1 = G_define_standard_option(G_OPT_R_INPUT);
  input1->key = "input_dem";
  output = G_define_standard_option(G_OPT_R_OUTPUT);

  /* Define the different flags */
  flag1 = G_define_flag();
  flag1->key = 'q';
  flag1->description = _("Quiet");


  opt7 = G_define_option();
  opt7->key = "loss_exp";
  opt7->type = TYPE_DOUBLE;
  opt7->required = NO;
  opt7->answer = "2.0";
  opt7->description = _("Exponent value (free-space-based fading model)");

  opt8 = G_define_option();
  opt8->key = "loss_offset";
  opt8->type = TYPE_DOUBLE;
  opt8->required = NO;
  opt8->answer = "0.0";
  opt8->description = _("Offset value [dB] (free-space-based fading model)");

  opt1 = G_define_option();
  opt1->key = "coordinate";
  opt1->type = TYPE_STRING;
  opt1->required = YES;
  opt1->key_desc = "x,y";
  opt1->description = _("Base station coordinates");

  opt6 = G_define_option();
  opt6->key = "radius";
  opt6->type = TYPE_DOUBLE;
  opt6->required = NO;
  opt6->answer = "10";
  opt6->description = _("Computation radius [km]");

  opt2 = G_define_option();
  opt2->key = "ant_height";
  opt2->type = TYPE_DOUBLE;
  opt2->required = NO;
  opt2->answer = "10";
  opt2->description = _("Transmitter antenna height [m]");

  opt5 = G_define_option();
  opt5->key = "rx_ant_height";
  opt5->type = TYPE_DOUBLE;
  opt5->required = NO;
  opt5->answer = "1.5";
  opt5->description = _("Receiver antenna height [m]");

  opt3 = G_define_option();
  opt3->key = "frequency";
  opt3->type = TYPE_DOUBLE;
  opt3->required = YES;
  opt3->description = _("Frequency [MHz]");


  /* options and flags parser */
  if (G_parser(argc, argv))
    exit( EXIT_FAILURE);

  /* stores options and flags to variables */
  name = input1->answer;
  result = output->answer;
  verbose = ( !flag1->answer);
  G_scan_easting( opt1->answers[0], &east, G_projection());
  G_scan_northing( opt1->answers[1], &north, G_projection());
  sscanf( opt2->answer, "%lf", &ant_height);
  sscanf( opt3->answer, "%lf", &frequency);
  sscanf( opt6->answer, "%lf", &radius);

  sscanf( opt5->answer, "%lf", &rec_height);
  sscanf( opt7->answer, "%lf", &loss_exp);
  sscanf( opt8->answer, "%lf", &loss_offset);


  G_get_window(&window);

  /* check if specified transmitter location inside window   */
  if ( east < window.west || east > window.east || north > window.north || north < window.south)
      G_fatal_error(_("Specified base station coordinates are outside current region bounds."));


  /* returns NULL if the map was not found in any mapset, mapset name otherwise */
  mapset = G_find_raster(name, "");
  if ( mapset == NULL)
    G_fatal_error(_("Raster map <%s> not found"), name);

  if (G_legal_filename(result) < 0)
    G_fatal_error(_("<%s> is an illegal file name"), result);

  /* Rast_open_old - returns file destriptor (>0) */
  if (( infd = Rast_open_old( name, mapset)) < 0)
    G_fatal_error(_("Unable to open raster map <%s>"), name);

  /* open input raster */
  Rast_get_cellhd( name, mapset, &cellhd);

  /* Allocate input buffer */
  inrast = Rast_allocate_buf( FCELL_TYPE);

  /* Allocate output buffer, use input map data_type */
  nrows = Rast_window_rows();
  ncols = Rast_window_cols();
  outrast = Rast_allocate_buf( FCELL_TYPE);

  /* controlling, if we can write the raster */
  if (( outfd = Rast_open_new( result, FCELL_TYPE)) < 0)
    G_fatal_error(_("Unable to create raster map <%s>"), result);

  /* map array coordinates for transmitter */
  tr_row = ( window.north - north) / window.ns_res;
  tr_col = ( east - window.west) / window.ew_res;

  // check if transmitter is on DEM
  FCELL trans_elev;
  Rast_get_row( infd, inrast, tr_row, FCELL_TYPE);
  trans_elev = ((FCELL *) inrast)[tr_col];
  if ( isnan( (double)trans_elev))
    G_fatal_error(_("Transmitter outside raster DEM map."));

  /*--- define structure variables----*/
  double BSAntHeight = ant_height;     
  double MSAntHeight = rec_height;     
  int xN = window.rows;
  int yN = window.cols;
  double scale = window.ew_res;
  double freq = frequency;
  double ResDist = 1;
  double BSyIndex = ( east - window.west) / scale - 0.5;
  double BSxIndex = ( window.north - north) / scale - 0.5;
  double radi = radius;

  struct StructFSPL IniFSPL = { BSxIndex, BSyIndex, BSAntHeight, MSAntHeight, xN, yN, scale, freq, ResDist, radi};


  /* allocate the memory to contain the whole file */
  /*RASTER*/
  double **m_rast;
  int i;
  m_rast = (double **)G_calloc( nrows, sizeof(double *));
  double *tmp_rast = (double *)G_calloc( nrows * ncols, sizeof(double));
  memset( tmp_rast, 0, nrows * ncols * sizeof(double));
  for ( i = 0 ; i < nrows ; i++) m_rast [i] = tmp_rast + i * ncols;

  /*PATH LOSS*/
  double **m_loss;
  int k;
  m_loss = (double **)G_calloc( nrows, sizeof(double *));
  double *tmp_loss = (double *)G_calloc( nrows * ncols, sizeof(double));
  memset ( tmp_loss, 0, nrows * ncols * sizeof(double));
  for ( k = 0; k < nrows ; k++) m_loss[k] = tmp_loss + k * ncols;


  /* Write file (raster) to array */
  /* for each row */
  for (row = 0; row < nrows; row++)
  { 
    if (verbose)
      G_percent( row, nrows, 2);

    /* read input map */
    Rast_get_row( infd, inrast, row, FCELL_TYPE);

    /* write to aray */
    for ( col = 0; col < ncols; col++)
      m_rast[row][col] = (double)((FCELL *) inrast)[col];
  }


  /* compute signal loss according to the FSPL model */
  FSPL_PathLossSub( m_rast, m_loss, IniFSPL, loss_exp, loss_offset);

    
  double path_loss_num;
  FCELL  null_f_out;

  Rast_set_f_null_value( &null_f_out, 1);   

  /* write to output raster, null_f_out is for transparente cells (no signal, NLOS) */
  for ( row = 0; row < nrows; row++)
  {
    G_percent( row, nrows, 2);
    for ( col = 0; col < ncols; col++) 
    {
      path_loss_num = m_loss[row][col];
      if ( path_loss_num == 0) ((FCELL *) outrast)[col] = null_f_out;
      else                     ((FCELL *) outrast)[col] = (FCELL)path_loss_num;
    }
    /* write raster row to output raster map */
    Rast_put_row( outfd, outrast, FCELL_TYPE);
  }


  /* memory cleanup */
  G_free( inrast);
  G_free( outrast);

  /* closing raster maps */
  Rast_close( infd);
  Rast_close( outfd);

  /* add command line incantation to history file */
  Rast_short_history( result, "raster", &history);
  Rast_command_history( &history);
  Rast_write_history( result, &history);


  exit( EXIT_SUCCESS);
}



int FSPL_PathLossSub( double** Raster, double** PathLoss, struct StructFSPL IniFSPL, double loss_exp, double loss_offset)
{

  // FSPL model constants and variables
  double BSxIndex = IniFSPL.BSxIndex;        // normalized position of BS -> UTMx/resolution 
  double BSyIndex = IniFSPL.BSyIndex;        // normalized position of BS -> UTMy/resolution
  double AntHeightBS = IniFSPL.BSAntHeight;  // antenna height of BS [m]
  double AntHeightMS = IniFSPL.MSAntHeight;  // antenna height of MS [m]
  int xN = IniFSPL.xN;                       // dimension of the input(Raster) and output (PathLoss)
  int yN = IniFSPL.yN;                       // dimension of the input(Raster) and output (PathLoss)
  double scale = IniFSPL.scale;              // resolution FSPL model
  double freq  = IniFSPL.freq;               // carrier frequency
  double ResDist = IniFSPL.ResDist;          // distance BS - MS sampling rate [normalized with scale
  double radi = IniFSPL.radi;                // radius of calculation

  double ZoBS;                               // BS and MS height about the sea level
  double ZObs2LOS = 0;
  double DistObs2BS = 0;
  double ZoTransBS,ZoTransMS;

  double tiltBS2MS;                          // ( ZoBS-ZoMS) / distBS2MSNorm

  double PathLossTmp = 0;                    // tmp path loss
  int ix; int iy;
  double DiffX, DiffY, Zeff;                 // Difference in X and Y direction
  double DistBS2MSNorm, DistBS2MSKm, R;      // distance between MS and BS in Km sqrt(x2+y2+z2) * scale / 1000
                                             // normalized distance between MS and BS in xy plan sqrt(x2+y2)

  ZoBS = Raster[(int)( BSxIndex + 0.5)][(int)( BSyIndex + 0.5)];  // BS height above the sea level calculated from raster DEM file
  ZoTransBS = ZoBS + AntHeightBS;            // BS transmitter height above the sea level

  for ( ix = 0; ix < xN; ix++)
  {
    G_percent( ix, xN, 2);
    for ( iy = 0; iy < yN; iy++)
    {
      DiffX = ( BSxIndex - ix); DiffY = ( BSyIndex - iy);
      ZoTransMS = Raster[ix][iy] + AntHeightMS;  // ZoMS
      Zeff = ZoTransBS - ZoTransMS;
      DistBS2MSNorm = sqrt( DiffX * DiffX + DiffY * DiffY);
      DistBS2MSKm = DistBS2MSNorm * scale / 1000.0;  //  Ground distance between Tx and Rx (in [km])

      if ( DistBS2MSKm > radi || DistBS2MSNorm == 0) continue;

      R = sqrt( pow( DistBS2MSKm, 2.0) + pow( Zeff / 1000.0, 2.0) ); // Real (physical) distance between Tx and Rx 

      if ( ZoBS <= Raster[ix][iy]) Zeff = AntHeightBS;  // ZoMS
      if ( DistBS2MSKm < 0.01) DistBS2MSKm = 0.01;


      // Calc position of the height and position of the highest obstacle
      tiltBS2MS = ZoTransBS - ZoTransMS; 
      if ( DistBS2MSNorm > 0) tiltBS2MS = -tiltBS2MS / DistBS2MSNorm;
      else tiltBS2MS = 0;
      DoProfile( &ZObs2LOS, &DistObs2BS, ResDist, Raster, BSxIndex, BSyIndex, ZoTransBS, ix, iy, tiltBS2MS);

      // Free space signal loss for LOS, 0 for NLOS
      if ( ZObs2LOS < 0) PathLossTmp = 32.4 + ( 10 * loss_exp) * log10( R) + 20 * log10( freq) + loss_offset; /* LOS */
      else PathLossTmp = 0;


      // write data to pathloss
      PathLoss[ix][iy] = PathLossTmp; 

      } // end irow

    } // end icol

    return 0;
  }

