/**
 * @author Urban Duh <urban.duh@student.fmf.uni-lj.si>
 * @author Gregor Kosec <gregor.kosec@ijs.si>
 * @author Jure Slak <jure.slak@ijs.si>
 * @date 2020-7-24
 * @brief Illustrates the surface node placing algorithm.
 */

#include <Eigen/Core>
#include "PA.hpp"
#include <fstream>
#include <iostream>
#include <cmath>

#define PI 3.1415926535897932385

typedef Eigen::Matrix<double, 1, 1> Vec1d;
typedef Eigen::Matrix<double, 2, 1> Vec2d;
typedef Eigen::Matrix<double, 3, 1> Vec3d;

void polar_2d() {
    auto example_r = [](const Vec1d& t) {
        t(0) = -t(0);
        double r = pow(abs(cos(1.5 * t(0))), sin(3 * t(0)));
        return Vec2d(r * cos(t(0)), r * sin(t(0)));
    };

    auto der_example_r = [](const Vec1d& t) {
        t(0) = -t(0);
        double r = pow(abs(cos(1.5 * t(0))), sin(3 * t(0)));
        double der_r = (-1.5 * pow(abs(cos(1.5 * t(0))),
                                   sin(3 * t(0))) * sin(3 * t(0)) * sin(1.5 * t(0)) +
                        3 * pow(abs(cos(1.5 * t(0))),
                                sin(3 * t(0))) * cos(3 * t(0)) * cos(1.5 * t(0))
                        * log(abs(cos(1.5 * t(0))))) / cos(1.5 * t(0));

        Eigen::Matrix<double, 2, 1> jm;
        jm.col(0) << -(der_r * cos(t(0)) - r * sin(t(0))),
                     -(der_r * sin(t(0)) + r * cos(t(0)));

        return jm;
    };

    auto is_inside_param = [](const Vec1d& t) {
        return t(0) >= 0.0 && t(0) <= 2 * PI;
    };
    auto h = [](const Vec2d& t) {
        return 0.04;
    };

    Vec1d start(PI / 2);
    int num_candidates = 2;
    int num_points = 1000000;
    std::mt19937 gen(1337);

    auto pts = pa<double, 2, 1>(is_inside_param, {start}, example_r, der_example_r,
        h, num_candidates, num_points, gen);

    int N = pts.size();
    std::cout << "N = " << N << std::endl;

    std::string filename = "example_2d.txt";
    std::ofstream file(filename);

    for (int i = 0; i < N; ++i) {
        for (int j = 0; j < 2; ++j)
            file << pts[i][j] << " ";
        file << "\n";
    }
}

void heart_3d() {
    auto heart_r = [](const Vec2d& t) {
        double R = sqrt(1 - t(1) * t(1));
        return Vec3d(R * cos(t(0)) + t(1)*t(1), R * sin(t(0)), t(1));
    };

    auto heart_jacobian_matrix = [](const Vec2d& t) {
        double R = sqrt(1 - t(1) * t(1));
        double der_R = - t(1) / sqrt(1 - t(1) * t(1));
        Eigen::Matrix<double, 3, 2> jm;
        jm.col(0) << - R * sin(t(0)), R * cos(t(0)), 0;
        jm.col(1) << der_R * cos(t(0)) + 2 * t(1), der_R * sin(t(0)), 1;
        return jm;
    };

    auto is_inside_param = [](const Vec2d& t) {
        return t(0) >= 0.0 && t(0) <= 2 * PI && t(1) >= -1.0 && t(1) <= 1.0;
    };
    auto h = [](const Vec3d& t) {
        return 0.05;
    };

    Vec2d start(PI, 0);
    int num_candidates = 15;
    int num_points = 1000000;
    std::mt19937 gen(1337);

    auto pts = pa<double, 3, 2>(is_inside_param, {start}, heart_r, heart_jacobian_matrix,
        h, num_candidates, num_points, gen);

    int N = pts.size();
    std::cout << "N = " << N << std::endl;

    std::string filename = "example_3d.txt";
    std::ofstream file(filename);

    for (int i = 0; i < N; ++i) {
        for (int j = 0; j < 3; ++j)
            file << pts[i][j] << " ";
        file << "\n";
    }
}

int main() {
    polar_2d();
    heart_3d();

    return 0;
}
