#ifndef FILLALGORITHMS_PA_HPP
#define FILLALGORITHMS_PA_HPP

#include <random>
#include <cassert>
#include <cmath>
#include <iostream>
#include <Eigen/Core>
#include <Eigen/StdVector>
#include "nanoflann.hpp"

/**
 * Standalone implementation of the surface node placing algorithm presented in the paper.
 * @tparam scalar_t Floating point number type, eg. `float` or `double`.
 * @tparam dim Dimension of the domain.
 * @tparam param_dim Dimension of the parametric domain.
 * @param is_inside_param Characteristic function of the parametric domain, which takes a point and returns `true` if
 * and only if given point is inside the domain.
 * @param starting_params List of starting parameters (from the parametric domain) for the algorithm.
 * @param param_function Parametrization function.
 * @param param_jacobian Jacobian matrix of the parametrization function.
 * @param spacing_function Function prescribing local nodal spacing, which takes a point and returns a positive
 * floating point number.
 * @param num_samples Controls the number of generated candidates from each point.
 * @param max_points Maximal number of points. The algorithm terminates once this threshold has been
 * reached. May generate up to `num_samples` more points than `max_points`.
 * @param generator (Pseudo) random generator to be used for randomization.
 */
template <typename scalar_t, int dim, int param_dim, typename spacing_func_t, typename param_domain_func_t,
typename param_func_t, typename jm_func_t, typename generator_t>
std::vector<Eigen::Matrix<scalar_t, dim, 1>,
Eigen::aligned_allocator<Eigen::Matrix<scalar_t, dim, 1>>> pa(param_domain_func_t is_inside_param,
        const std::vector<Eigen::Matrix<scalar_t, param_dim, 1>,
        Eigen::aligned_allocator<Eigen::Matrix<scalar_t, param_dim, 1>>> starting_params,
        const param_func_t& param_function, const jm_func_t& param_jacobian,
        const spacing_func_t& spacing_function, int num_samples, size_t max_points,
        generator_t& generator);

/// Implementation details of PA.
namespace pa_impl {

template <typename T>
struct Pi {
    static constexpr T value = T(3.1415926535897932385L);
};

/**
 * Discretizes a sphere with given radius uniformly with `num_points` points on the great circle.
 * This is in a class because C++ does not allow partial specializations of template functions.
 * @tparam dim Dimension of the sphere.
 * @tparam scalar_t Data type for numeric computations, e.g. `double` or `float`.
 * @param radius Radius of the sphere.
 * @param num_samples Number of points on the equator, implies nodal spacing `dp = 2*pi*r/n`.
 * @param generator A random generator.
 * @return A vector of discretization points.
 */
template <typename scalar_t, int dim>
struct Sphere {
    template <typename generator_t>
    static std::vector<Eigen::Matrix<scalar_t, dim, 1>,
                       Eigen::aligned_allocator<Eigen::Matrix<scalar_t, dim, 1>>> discretize(
            scalar_t radius, int num_samples, generator_t& generator) {
        scalar_t dphi = 2 * Pi<scalar_t>::value / num_samples;
        std::uniform_real_distribution<scalar_t> distribution(0, Pi<scalar_t>::value);
        scalar_t offset = distribution(generator);
        std::vector<Eigen::Matrix<scalar_t, dim, 1>,
                Eigen::aligned_allocator<Eigen::Matrix<scalar_t, dim, 1>>> result;
        for (int i = 0; i < num_samples / 2; ++i) {
            scalar_t phi = i * dphi + offset;
            if (phi > Pi<scalar_t>::value) phi -= Pi<scalar_t>::value;
            int slice_n = static_cast<int>(std::ceil(num_samples * std::sin(phi)));
            if (slice_n == 0) continue;
            auto slice = Sphere<scalar_t, dim - 1>::discretize(
                    radius * std::sin(phi), slice_n, generator);
            Eigen::Matrix<scalar_t, dim, 1> v;
            for (const auto& p : slice) {
                v[0] = radius * std::cos(phi);
                v.template tail<dim - 1>() = p;
                result.push_back(v);
            }
        }
        return result;
    }
};

/// Two-dimensional base case of the discretisation.
template <typename scalar_t>
struct Sphere<scalar_t, 2> {
    template <typename generator_t>
    static std::vector<Eigen::Matrix<scalar_t, 2, 1>,
            Eigen::aligned_allocator<Eigen::Matrix<scalar_t, 2, 1>>> discretize(
            scalar_t radius, int num_samples, generator_t& generator) {
        scalar_t dphi = 2 * Pi<scalar_t>::value / num_samples;
        std::uniform_real_distribution<scalar_t> distribution(0, 2 * Pi<scalar_t>::value);
        scalar_t offset = distribution(generator);
        std::vector<Eigen::Matrix<scalar_t, 2, 1>,
                Eigen::aligned_allocator<Eigen::Matrix<scalar_t, 2, 1>>> result;
        for (int i = 0; i < num_samples; ++i) {
            scalar_t phi = i * dphi + offset;
            result.emplace_back(radius * std::cos(phi), radius * std::sin(phi));
        }
        return result;
    }
};

/// One-dimensional base case of the discretisation.
template <typename scalar_t>
struct Sphere<scalar_t, 1> {
    template <typename generator_t>
    static std::vector<Eigen::Matrix<scalar_t, 1, 1>,
            Eigen::aligned_allocator<Eigen::Matrix<scalar_t, 1, 1>>> discretize(
            scalar_t radius, int, generator_t&) {
        return {Eigen::Matrix<scalar_t, 1, 1>(-radius), Eigen::Matrix<scalar_t, 1, 1>(radius)};
    }
};


/// Helper class for KDTree with appropriate accessors containing a set of points.
template <typename scalar_t, int dim>
struct PointCloud {
    typedef typename std::vector<Eigen::Matrix<scalar_t, dim, 1>,
                                 Eigen::aligned_allocator<Eigen::Matrix<scalar_t, dim, 1>>>
                                 container_t;
    container_t data;  ///< Points contained in the tree.

    /// Construct an empty point set.
    PointCloud() = default;

    /// Construct from an array of points.
    PointCloud(container_t pts) : data(std::move(pts)) {}

    /// Interface requirement: returns number of data points.
    inline int kdtree_get_point_count() const { return static_cast<int>(data.size()); }

    /// Interface requirement: returns `d`-th coordinate of `idx`-th point.
    inline scalar_t kdtree_get_pt(const size_t idx, int d) const {
        return data[idx][d];
    }

    /// Comply with the interface.
    template <class BBOX>
    bool kdtree_get_bbox(BBOX& /* bb */) const { return false; }
};


}  // namespace pnp_impl

/// Implementation of the proposed algorithm.
template <typename scalar_t, int dim, int param_dim, typename spacing_func_t, typename param_domain_func_t,
typename param_func_t, typename jm_func_t, typename generator_t>
std::vector<Eigen::Matrix<scalar_t, dim, 1>,
Eigen::aligned_allocator<Eigen::Matrix<scalar_t, dim, 1>>> pa(param_domain_func_t is_inside_param,
        const std::vector<Eigen::Matrix<scalar_t, param_dim, 1>,
        Eigen::aligned_allocator<Eigen::Matrix<scalar_t, param_dim, 1>>> starting_params,
        const param_func_t& param_function, const jm_func_t& param_jacobian,
        const spacing_func_t& spacing_function, int num_samples, size_t max_points,
        generator_t& generator) {
    typedef Eigen::Matrix<scalar_t, dim, 1> vec_t;
    typedef Eigen::Matrix<scalar_t, param_dim, 1> param_vec_t;

    typedef nanoflann::KDTreeSingleIndexDynamicAdaptor<
    nanoflann::L2_Simple_Adaptor<scalar_t, pa_impl::PointCloud<scalar_t, dim>>,
                                 pa_impl::PointCloud<scalar_t, dim>, dim> kd_tree_t;

     size_t cur_node = 0;
     size_t end_node = starting_params.size();
     double zeta = 1-1e-10;
     assert(cur_node < end_node && "At least one start node must be given");

     pa_impl::PointCloud<scalar_t, param_dim> params(starting_params);
     pa_impl::PointCloud<scalar_t, dim> points;
     points.data.resize(starting_params.size());
     for (int i = 0; i < static_cast<int>(starting_params.size()); i++) {
         points.data[i] = param_function(starting_params[i]);
     }

     kd_tree_t tree(dim, points, nanoflann::KDTreeSingleIndexAdaptorParams(20));

    // Main algorithm loop.
    while (cur_node < end_node && end_node < max_points) {
        param_vec_t param_point = params.data[cur_node];
        auto jm = param_jacobian(param_point);
        vec_t initial_pt = param_function(param_point);

        auto unit_candidates = pa_impl::Sphere<scalar_t, param_dim>::
                               discretize(1, num_samples, generator);

        // Filter unit_candidates regarding the domain and proximity criteria.
        for (const auto& u_cand : unit_candidates) {
            scalar_t alpha = spacing_function(initial_pt) / (jm * u_cand).norm();
            param_vec_t node = param_point + alpha * u_cand;
            if (!is_inside_param(node)) continue;

            auto new_pt = param_function(node);
            nanoflann::KNNResultSet<scalar_t, int> resultSet(1);
            int neighbor_index;
            scalar_t dist_sq;
            resultSet.init(&neighbor_index, &dist_sq);
            tree.findNeighbors(resultSet, new_pt.data(), nanoflann::SearchParams(1));

            // Check radius must be new radius, otherwise algorithm might terminate in 2d.
            scalar_t check_radius = (new_pt - initial_pt).norm();
            // Check if node is far enough from other nodes.
            if (dist_sq < (zeta * check_radius) * (zeta * check_radius)) continue;

            // Insert into domains
            params.data.push_back(node);
            points.data.push_back(new_pt);
            // Update kd tree
            tree.addPoints(end_node, end_node);
            end_node++;
        }
        cur_node++;
    }
    if (end_node >= max_points) {
        std::cerr << "Maximum number of points reached, fill may be incomplete." << std::endl;
    }

    return points.data;
}

#endif  // PA_HPP
